/***********************************************************************/
/**
 \file
 Functions for reading DSP code from file asidsp.bin using HpiOs_ file functions

Copyright (C) 1997-2017 AudioScience, Inc. All rights reserved.

This software is provided 'as-is', without any express or implied warranty.
In no event will AudioScience Inc. be held liable for any damages arising
from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.
3. This copyright notice and list of conditions may not be altered or removed
   from any source distribution.

AudioScience, Inc. <support@audioscience.com>

( This license is GPL compatible see http://www.gnu.org/licenses/license-list.html#GPLCompatibleLicenses )

*/
/***********************************************************************/
#define SOURCEFILE_NAME "hpidspcd.c"
#include "hpidspcd.h"
#include "hpidebug.h"
#include "hpi_version.h"

#ifndef HPI_BUILD_KERNEL_MODE
#include <string.h>
#include <stdlib.h>
#endif

/* must match or be greater than the same constant in S2BIN.C */
#define BLOCK_LIMIT_DWORD 2048L


#ifndef DISABLE_PRAGMA_PACK1
#pragma pack(push, 1)
#endif

#ifndef DISABLE_PRAGMA_PACK1
#pragma pack(pop)
#endif

/** state specific to this loader */
struct dsp_code_private {
	/// File descriptor for dsp code file
	HpiOs_FILE file;
	/// dsp code buffer
	uint32_t *buffer;
};

/********************************************************************/
/* Load dspcode from file asidsp.bin */


/*-------------------------------------------------------------------*/

short HpiDspCode_Open(uint32_t nAdapter, struct hpi_os_adapter *pOsData,
		      struct dsp_code *dsp_code)
{
	hpi_err_t err;
	int i, nRead = 0;
	struct code_header header;
	const char *pszFilepath;
	HpiOs_FILE file;

	pszFilepath = HpiOs_GetDspCodePath(nAdapter);
	HPI_DEBUG_LOG1(DEBUG, "Loading DSP code file: \"%s\"\n",pszFilepath);

	err = HpiOs_fopen_rb(pszFilepath, &file);
	if (err)
		return (err);

	memset(&header, 0, sizeof(header));
	// seek for first CODE header
	for (i = 0; i < 1000; i++) {
		HpiOs_fseek(file, i, SEEK_SET);
		nRead = (int)HpiOs_fread(&header, 1, sizeof(header),
				file);
		if (nRead < sizeof(header)) {
			HPI_DEBUG_LOG0(DEBUG, "Ran out of data\n");
			break;
		}

		if (header.type == 0x45444F43)
			break;
	}
	if ((nRead < sizeof(header)) || (i == 1000)) {
		HpiOs_fclose(file);
		HPI_DEBUG_LOG0(DEBUG, "CODE header not found\n");
		return HPI_ERROR_DSP_FILE_NO_HEADER;
	}
	// walk CODE headers searching for correct one
	while ((nRead == sizeof(header)) && (header.adapter != nAdapter)) {
		HPI_DEBUG_LOG1(VERBOSE, "Skipping %04x ",
			       header.adapter);
		if (header.size == 0) {
			HpiOs_fclose(file);
			HPI_DEBUG_LOG0(DEBUG,
				       "NULL header found\n");
			return HPI_ERROR_DSP_FILE_NULL_HEADER;
		}
		HpiOs_fseek(file,
			    (header.size - sizeof(header)), SEEK_CUR);
		nRead =(int)HpiOs_fread(&header, 1, sizeof(header),
				file);
	}
	if ((nRead < sizeof(header)) || (header.adapter != nAdapter)) {
		HpiOs_fclose(file);
		HPI_DEBUG_LOG1(ERROR,
			       "Adapter type %04x not found\n",
			       nAdapter);
		return HPI_ERROR_DSP_SECTION_NOT_FOUND;
	}
	else
	{
		HPI_DEBUG_LOG1(VERBOSE,
			       "Adapter type %04x code found\n",
			       nAdapter);
	}

	dsp_code->pvt = HpiOs_MemAlloc(sizeof(*dsp_code->pvt));
	if (!dsp_code->pvt)
		return HPI_ERROR_MEMORY_ALLOC;

	dsp_code->pvt->buffer =
	    HpiOs_MemAlloc(sizeof(uint32_t) * BLOCK_LIMIT_DWORD);
	if (!dsp_code->pvt->buffer) {
		HpiOs_fclose(dsp_code->pvt->file);
		HPI_DEBUG_LOG0(ERROR, "Out of memory error\n");
		HpiOs_MemFree(dsp_code->pvt);
		return HPI_ERROR_MEMORY_ALLOC;
	}
	dsp_code->pvt->file = file;
	dsp_code->header = header;
	dsp_code->block_length = (header.size - sizeof(header)) / 4;
	dsp_code->word_count = 0;
	return 0;
}

/*-------------------------------------------------------------------*/
void HpiDspCode_Rewind(struct dsp_code *dsp_code)
{
	/* why not rewind and re-seek? */
	HpiDspCode_Close(dsp_code);
	HpiDspCode_Open(dsp_code->header.adapter, NULL, dsp_code);
}

/*-------------------------------------------------------------------*/
void HpiDspCode_Close(struct dsp_code *dsp_code)
{
	HpiOs_MemFree(dsp_code->pvt->buffer);
	HpiOs_fclose(dsp_code->pvt->file);
	HpiOs_MemFree(dsp_code->pvt);
}

/*-------------------------------------------------------------------*/
short HpiDspCode_ReadWord(struct dsp_code *dsp_code, uint32_t *pdwWord)
{
	size_t count;
	count = HpiOs_fread(pdwWord, sizeof(uint32_t), 1, dsp_code->pvt->file);
	if (count != 1)
		return (HPI_ERROR_DSP_FILE_FORMAT + 21);	// 231
	// could also calculate checksum here?
	dsp_code->word_count++;
	if (dsp_code->word_count > dsp_code->block_length)
		return (HPI_ERROR_DSP_FILE_FORMAT + 22);	// 232
	return 0;
}

/*-------------------------------------------------------------------*/
short HpiDspCode_ReadBlock(size_t nWordsRequested,
			   struct dsp_code *dsp_code, uint32_t **ppdwBlock)
{
	size_t count;

	if (nWordsRequested > BLOCK_LIMIT_DWORD)
		return (HPI_ERROR_DSP_FILE_FORMAT + 23);	// 233

	count = HpiOs_fread(dsp_code->pvt->buffer, sizeof(uint32_t), nWordsRequested,
			dsp_code->pvt->file);
	if (count != nWordsRequested)
		return (HPI_ERROR_DSP_FILE_FORMAT + 24);	// 234

	dsp_code->word_count += (long)nWordsRequested;
	if (dsp_code->word_count > dsp_code->block_length)
		return (HPI_ERROR_DSP_FILE_FORMAT + 25);	// 235

	*ppdwBlock = dsp_code->pvt->buffer;
	return (0);
}
